/*  ------------------------------------------------------------------
    Unimodem Diagnostics, copyright (c) 2000-2002 Stanislav V. Mekhanoshin
    ------------------------------------------------------------------
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    ------------------------------------------------------------------
    $Id: #ud.c, v 1.1.0.1 $
    ------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <limits.h>
#include <malloc.h>
#include <ctype.h>
#include <io.h>
#include <share.h>
#include <errno.h>
#ifdef DJGPP
#include <unistd.h>
#endif
#ifdef __RSXNT__
#include <alloca.h>
#endif

#if defined(__NT__) || defined(_WIN32) || defined(__RSXNT__) || defined(_WINDOWS)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#if !defined(_fsopen) && !defined(__RSXNT__) && defined(__GNUC__)
#define _fsopen(f,m,s) fopen(f,m)
#endif

#if defined(_WINDOWS)
#ifdef _MSC_VER
#pragma warning(push,1)
#endif /* _MSC_VER */
#include <commdlg.h>
#include <shellapi.h>
#include <commctrl.h>
#ifdef _MSC_VER
#pragma warning(pop)
#endif /* _MSC_VER */
#include "resource.h"
#include "resrc1.h"
#endif /* _WINDOWS */

#if defined(__RSXNT__)
#define __PLATFORM__ RSX
#define __CLIPBOARD__
#define IsW32 ((_emx_env & 0x1000) && ((_emx_rev>>16) == 2))
#elif defined(__DOS__)
#define __PLATFORM__ DOS
#elif defined(__OS2__)
#define __PLATFORM__ 2
#elif defined(__NT__) || defined(_WIN32)
#define __PLATFORM__ W32
#define __CLIPBOARD__
#define IsW32 (1)
#elif defined(DJGPP)
#define __PLATFORM__ DJG
#define __cdecl
#elif defined(__unix__) || defined(__UNIX__)
#define __cdecl
#define __PLATFORM__ *NIX
#else
#error Undefined platform!
#endif

#define __string(p) #p
#define __value(p)  __string(p)
#define __PLATFORM  __value(__PLATFORM__)

static const char Logo1[]="Unimodem Diagnostics/"__PLATFORM", version 1.1.0.1, compiled at "__DATE__" "__TIME__;
static const char Logo2[]="(c) 2000 Stanislav V. Mekhanoshin (rampitec@tu.spb.ru, 2:5030/172.9@fidonet)";
static const char Logo3[]="----------------------------------------------------------------------------";

#define P_NO   USHRT_MAX
#define P_XX   (USHRT_MAX-1)

#pragma pack(1)

typedef enum _Method{
    M_STRING,   /* print out string */
    M_TABLE,    /* lookup table     */
    M_VALUE,    /* print out value  */
    M_SPECIAL   /* special case     */
}Method;

typedef struct _Table{
    unsigned short      code;
    const char* const   desc;
}Table;

typedef struct _Directory{
    const Method         meth;
    const Table* const   table;
    const char* const    desc;
    const unsigned short code1;
    const unsigned short code2;
    const unsigned short max1;   /* max value1 for %d */
    const unsigned short max2;   /* max value2 for %d */
}Directory;

static const Table table2[]={
{0x0, "No previous call"},
{0x1, "No dial tone detected"},
{0x2, "Reorder signal detected, network busy"},
{0x3, "Busy signal detected"},
{0x4, "No recognized signal detected"},
{0x5, "Voice detected"},
{0x6, "Text telephone signal detected"},
{0x7, "Data Answering signal detected"},
{0x8, "Data Calling signal detected"},
{0x9, "Fax Answering signal detected"},
{0xA, "Fax Calling signal detected"},
{0xB, "V.8bis signal detected"},
{P_NO, NULL}};

static const Table table3[]={
{0x0, "Data Only"},
{0x1, "FAX Only"},
{0x2, "Voice Only"},
{0x3, "VoiceViewT"},
{0x4, "ASVD, V.61"},
{0x5, "ASVD, <V.34Q>"},
{0x6, "DSVD, Multi-Tech"},
{0x7, "DSVD, 1.2"},
{0x8, "DSVD, V.70"},
{0x9, "Video-telephony, H.324"},
{0xA, "Other V.80 call"},
{P_NO, NULL}};

static const Table table4[]={
{0x0, "Async data"},
{0x1, "V.80 transparent synchronous mode"},
{0x2, "V.80 framed synchronous mode"},
{P_NO, NULL}};

static const Table table6[]={
{0x00, "V.17 (G3 Fax call)"},
{0x01, "V.21"},
{0x02, "V.22"},
{0x03, "V.22bis"},
{0x04, "V.23 Constant Carrier (1200/75)"},
{0x05, "V.23 Switched Carrier (half duplex)"},
{0x06, "V.26bis"},
{0x07, "V.26ter"},
{0x08, "V.27ter (G3 Fax call)"},
{0x09, "V.29 HD (G3 Fax call)"},
{0x0A, "V.32"},
{0x0B, "V.32bis"},
{0x0C, "V.34"},
{0x0D, "V.34 HD (G3 Fax call)"},
{0x0E, "V.90 Issue 1 (asymmetric)"},
{0x0F, "V.90 Issue 2 (symmetric)"},
{0x80, "<X2T>"},
{0x81, "<K56FLEXT>"},
{0x82, "<V.FC>"},
{0x83, "<V.32terbo>"},
{0x84, "Bell 212A"},
{0x85, "Bell 103"},
{P_NO, NULL}};

static const Table table7[]={
{0x00, "Disable/none"},
{0x01, "V.42 LAPM"},
{0x02, "V.42 Alternative protocol (MNPT)"},
{0x80, "MNP10T"},
{0x81, "ECPT Enhanced Cellular Protocol"},
{0x82, "ETCT Enhanced Throughput Cellular"},
{P_NO, NULL}};

static const Table table8[]={
{0x00, "None"},
{0x01, "V.42bis"},
{0x80, "MNP5T"},
{P_NO, NULL}};

static const Table table9[]={
{0x00, "Cause Unidentified"},
{0x01, "No Previous call"},
{0x02, "Call is still in progress"},
{0x03, "Call Waiting signal detected"},
{0x04, "Delayed"},
{0x0A, "NMS initiated Dial Call"},
{0x0B, "NMS initiated Leased Line Restoral"},
{0x0C, "NMS initiated Redial"},
{0x0D, "NMS initiated Dial Disconnect"},
{0x14, "Power Loss"},
{0x15, "Equipment Failure"},
{0x16, "Front Panel Disconnect Requested"},
{0x17, "Front Panel Leased Line Restoral"},
{0x18, "Automatic Leased Line Restoral"},
{0x19, "Inactivity Timer Expired"},
{0x1F, "cct108 is Off Inhibits Dial"},
{0x20, "cct108 turned Off"},
{0x28, "No Number Provided"},
{0x29, "Blacklisted Number"},
{0x2A, "Call Attempts Limit Exceeded"},
{0x2B, "Extension Phone Off Hook"},
{0x2C, "Call Setup Fail Timer Expired"},
{0x2D, "Incoming Call Detected"},
{0x2E, "Loop Current Interrupted"},
{0x2F, "No Dial Tone"},
{0x30, "Voice Detected"},
{0x31, "Reorder Tone"},
{0x32, "Sit Tone"},
{0x33, "Engaged Tone"},
{0x34, "Long Space Disconnect"},
{0x3C, "Carrier Lost"},
{0x3D, "Training Failed"},
{0x3E, "No Modulationin Common"},
{0x3F, "Retrain Failed"},
{0x40, "Retrain Attempt Count Exceeded"},
{0x41, "Gstn Cleardown Received"},
{0x42, "Fax Detected"},
{0x46, "In Test Mode"},
{0x47, "Intrusive Self Test Initiated"},
{0x50, "Any Key Abort"},
{0x51, "Dte Hangup Command"},
{0x52, "Dte Reset Command"},
{0x5A, "Frame Reject"},
{0x5B, "No Error Control Established"},
{0x5C, "Protocol Violation"},
{0x5D, "n400 exceeded"},
{0x5E, "Negotiation Failed"},
{0x5F, "Disconnect Frame Received"},
{0x60, "Sabme Frame Received"},
{0x64, "Loss Of Synchronization"},
{P_NO, NULL}};

static const Table add1[]={
{0x0, "off"},
{0x1, "DC1/DC3"},
{0x2, "V.24 ckt 106/133"},
{P_NO, NULL}};

static const char sNA[]="NA";
static const char sON[]="ON";
static const char sOFF[]="OFF";
static const char sPowerDropSupport[]="Remote Power Drop support";
static const char sFreqOffset[]="Freq offset Caller/Answer, Hz";
static const char sPreempIndex[]="Preemphasis Index Caller/Answer";
static const char sRateAsymmetry[]="V.34 Rate Asymmetry";

static const Directory table1[]={
{M_SPECIAL, 0,      "Diag Command Specification rev.", 0x00, P_NO},
{M_TABLE,   table2, "Call Setup Result code",          0x01, P_NO},
{M_TABLE,   table3, "Multi-media mode",                0x02, P_NO},
{M_TABLE,   table4, "DTE-DCE interface mode",          0x03, P_NO},
{M_STRING,  0,      "V.8 CM/JM octet string",          0x04, 0x05},
{M_VALUE,   0,      "TX/RX signal power level, -dBm",  0x11, 0x10, 0x001Fu, 0x002Fu},
{M_VALUE,   0,      "Estimated noise level, -dBm",     0x12, P_NO, 0x0064u, 0x0000u},
{M_VALUE,   0,      "Normalized Mean Squared error",   0x13, P_NO, 0x00FFu, 0x0000u},
{M_VALUE,   0,      "Near/Far echo loss, dB",          0x14, 0x15, 0x003Fu, 0x003Fu},
{M_VALUE,   0,      "Round Trip/Far echo delay, ms",   0x17, 0x16, 0x0FFFu, 0x003Fu},
{M_SPECIAL, 0,      NULL/*"V.34 INFO bit map"*/,       0x18, P_NO},
/* special keys: INFO0 decoding... */
{M_SPECIAL, 0,      sPowerDropSupport,                 P_XX, P_NO},
{M_SPECIAL, 0,      sFreqOffset,                       P_XX, P_XX},
{M_SPECIAL, 0,      sPreempIndex,                      P_XX, P_XX},
{M_SPECIAL, 0,      sRateAsymmetry,                    P_XX, P_NO},
/* end of INFO0 */
{M_TABLE,   table6, "TX/RX Negotiation",               0x20, 0x21},
{M_VALUE,   0,      "TX/RX Symbol Rate",               0x22, 0x23, 0x1F40u, 0x1F40u},
{M_VALUE,   0,      "TX/RX Carrier frequency, Hz",     0x24, 0x25, 0x0FA0u, 0x0FA0u},
{M_VALUE,   0,      "TX data rate (Last/Init)",        0x34, 0x26, 0xFA00u, 0xFA00u},
{M_VALUE,   0,      "RX data rate (Last/Init)",        0x35, 0x27, 0xFA00u, 0xFA00u},
{M_VALUE,   0,      "Temporary carrier loss count",    0x30, P_NO, 0x00FFu, 0x0000u},
{M_VALUE,   0,      "Carrier Rate Re-neg count",       0x31, P_NO, 0x00FFu, 0x0000u},
{M_VALUE,   0,      "Retrains Requested/Granted",      0x32, 0x33, 0x00FFu, 0x00FFu},
{M_VALUE,   0,      "Renegs Requested/Granted",        0x36, 0x37, 0x00FFu, 0x00FFu},
{M_SPECIAL, 0,      "Protocol/Compression",            0x40, 0x44},
{M_VALUE,   0,      "Error control frame size, bytes", 0x41, P_NO, 0x0400u, 0x0000u},
{M_VALUE,   0,      "Error control timeouts in TX",    0x42, P_NO, 0x00FFu, 0x0000u},
{M_VALUE,   0,      "Error control NAKs received",     0x43, P_NO, 0x00FFu, 0x0000u},
{M_VALUE,   0,      "Compression dict. size, bytes",   0x45, P_NO, 0x0800u, 0x0000u},
{M_TABLE,   add1,   "TX/RX flow control",              0x50, 0x51},
{M_VALUE,   0,      "TX/RX chars sent",                0x52, 0x53},
{M_VALUE,   0,      "TX/RX chars lost (data overrun)", 0x54, 0x55, 0xFFFFu, 0xFFFFu},
{M_VALUE,   0,      "TX/RX I-Frame count",             0x56, 0x57},
{M_VALUE,   0,      "TX/RX I-Frame error count",       0x58, 0x59, 0xFFFFu, 0xFFFFu},
{M_TABLE,   table9, "Termination Cause",               0x60, P_NO},
{M_VALUE,   0,      "Call Waiting event count",        0x61, P_NO, 0x00FFu, 0x0000u},
{M_STRING,  0,      "Manufacturer key",                0xFF, P_NO}
};

#define IsManufacturer(key,code) ((key)<=0xFF && (key)>=0x80 && (code)==0xFF)

static const char* value_from_table(const Table* table, unsigned long key)
{
    int i;

    for( i=0; table[i].code!=P_NO; i++ )
        if( table[i].code==key )
            return table[i].desc;
    return sNA;
}

#define BUFF_SIZE 512

#if !defined(_WINDOWS) || defined(_NO_COM)
static
#endif
char* prefix=0;

#ifdef _WINDOWS
static HWND hwe;
static const char crlf[] = "\r\n";
#define INIT_EDIT_BUFF_SIZE 4096
#ifdef _NO_COM
static
#endif
char *edit_buff;
#endif /*_WINDOWS */

static void println(const char* s)
{
#ifdef _WINDOWS
    DWORD l=strlen(edit_buff)+strlen(prefix)+strlen(s)+3;
    HANDLE hHeap=GetProcessHeap();
    if( HeapSize(hHeap,0,edit_buff)<l )
        edit_buff=HeapReAlloc(hHeap,HEAP_GENERATE_EXCEPTIONS,edit_buff,l);
    strcat(edit_buff,prefix);
    strcat(edit_buff,s);
    strcat(edit_buff,"\r\n");
#else /* _WINDOWS */
    printf( "%s%s\n", prefix, s );
#endif /* !_WINDOWS */
}

#if defined(_WINDOWS) && !defined(_NO_COM)
extern int isOleCall;
__inline int isdevice(FILE* hIn){ (void)hIn; return 0; }
/* #define stricmp(s1,s2) lstrcmpi(s1,s2) */
#else
#define isdevice(hIn) isatty(fileno(hIn))
#endif

#if !defined(_WINDOWS) || defined(_NO_COM)
static
#endif /* _WINDOWS */
int MakeStat(FILE* hIn)
{
    unsigned long key, i;
    int j, maxdesc=0;
    char *p, *e;
    int pnum;
    char* formal[sizeof(table1)/sizeof(table1[0])][2];
    int first=0;
    static const char sDiag1[]="DIAG <2A4D3263 ";
    static const char sDiag2[]="DIAG<2A4D3263 ";
#if defined(__CLIPBOARD__)
    char* buff;
#else
    char buff[BUFF_SIZE];
#endif
    char rep[128];

    for( i=0; i<sizeof(table1)/sizeof(table1[0]); i++ ){
        formal[i][0]=formal[i][1]=(char*)sNA;
        if( table1[i].desc ){
            j=strlen(table1[i].desc);
            if( j>maxdesc )maxdesc=j;
        }
    }

#if defined(__CLIPBOARD__)
#if defined(_WINDOWS) && !defined(_NO_COM)
    if( isOleCall ){
        strcpy(edit_buff,prefix);
        strcat(edit_buff,Logo1);
        strcat(edit_buff,crlf);
        strcat(edit_buff,prefix);
        strcat(edit_buff,Logo2);
        strcat(edit_buff,crlf);
        strcat(edit_buff,prefix);
        strcat(edit_buff,Logo3);
        strcat(edit_buff,crlf);
        buff=strstr((const char*)hIn,sDiag1);
        if( !buff )buff=strstr((const char*)hIn,sDiag2);
        if( !buff ){
            println( "no statistics data available!" );
            return 1;
        }
        hIn=NULL;
        goto RLOOP;
    }else
#endif /* _WINDOWS + COM */
    if( !hIn || isdevice(hIn) ){
        HANDLE hClip;
        if( !OpenClipboard(NULL) ){
            println( "cannot open clipboard!" );
            return 1;
        }
        hClip=GetClipboardData(CF_TEXT);
        if( !hClip ){
            println( "no text data in clipboard available!" );
            CloseClipboard();
            return 1;
        }
        p=strstr((const char*)hClip,sDiag1);
        if( !p )p=strstr((const char*)hClip,sDiag2);
        if( !p ){
            println( "no statistics data in clipboard available!" );
            CloseClipboard();
            return 1;
        }
        i=strlen(p)+sizeof(buff[0]);
        if( i<BUFF_SIZE )i=BUFF_SIZE;
        buff=(char*)alloca(i);
        if( !buff ){
            println( "stack overflow!" );
            CloseClipboard();
            return 1;
        }
        strcpy(buff,p);
        CloseClipboard();
        goto RLOOP;
    }else buff=(char*)alloca(BUFF_SIZE);
#endif /* __CLIPBOARD__ */

    while(
#if defined(__CLIPBOARD__)
        hIn && !isdevice(hIn) &&
#endif /* __CLIPBOARD__ */
        fgets(buff,BUFF_SIZE,hIn)
         ){
        char* nxt_after;
#if defined(__CLIPBOARD__)
RLOOP:
#endif /* __CLIPBOARD__ */
        nxt_after=&buff[-1];
        while( nxt_after ){
            char* nextln=&nxt_after[1];
            while(isspace((int)*nextln))nextln++;
            {
                /* process the windows log file prefix: 02-05-2002 23:19:59.30 - Recv: */
                int i;
                static const char pattern[]="dd.dd.dddd dd:dd:dd.dd - Recv: ";
                for( i=0; i<sizeof(pattern)-1 && nextln[i]; i++ ){
                    switch( pattern[i] ){
                    case 'd':
                        if( !isdigit((int)nextln[i]) ) goto NonWin;
                        break;
                    case '.':
                        if( !ispunct((int)nextln[i]) ) goto NonWin;
                        break;
                    default:
                        if( pattern[i]!=nextln[i] ) goto NonWin;
                    }
                }
                nextln+=sizeof(pattern)-1;
            }
NonWin:     i=strcspn(nextln,">\n\r"); /* 1st line delim */
            nxt_after=&nextln[i];
            if( !*nxt_after )nxt_after=NULL;
            else *nxt_after='\0';
            i=strlen(nextln);
            if( !i-- )continue;
            while(i && isspace((int)nextln[i]))nextln[i]='\0';
            if( first && strncmp(nextln,"OK",2)==0 )goto Alez;
            if( strncmp(nextln,sDiag1,sizeof(sDiag1)-1)==0 )
                p=&nextln[sizeof(sDiag1)-1];
            else if( strncmp(nextln,sDiag2,sizeof(sDiag2)-1)==0 )
                p=&nextln[sizeof(sDiag2)-1];
            else continue;
            first=1;
			e=0;
			for( j=0; p && p[j]; j++ ){
				if( isspace((int)p[j]) || p[j]=='>' || p[j]==':' ){
					p[j++]=0;
					while( isspace((int)p[j]) || p[j]=='>' || p[j]==':' ) j++;
					e=&p[j];
					if( !*e ) e=0;
					break;
				}
				if( p[j]=='\"' ){
					do{ j++; } while( p[j]!='\"' && p[j] );
					if( !p[j] ) break;
				}
			}
			if( j==0 ) p=0;
            while(p){
                key=strtoul(p,&p,16);
                for( i=0; i<sizeof(table1)/sizeof(table1[0]); i++ ){
                    if( table1[i].code1==key || IsManufacturer(key,table1[i].code1) ){
                        pnum=0;
                        goto Eq;
                    }
                    if( table1[i].code2==key ){
                        pnum=1;
                        goto Eq;
                    }
                }
                sprintf( rep, "unrecognized key %lx!", key );
                println(rep);
                goto Next;
Eq:             while( p && !*p ){
	                p=e;
					e=0;
					for( j=0; p && p[j]; j++ ){
						if( isspace((int)p[j]) ){
							p[j++]=0;
							while( isspace((int)p[j]) ) j++;
							e=&p[j];
							if( !*e ) e=0;
							break;
						}
					}
					if( j==0 ) p=0;
				}
                if( !p || *p!='=' ){
Uf:                 sprintf( rep, "unrecognized format '%s'!", nextln );
                    println(rep);
                    goto NextLn;
                }
                p++;
                while( !*p ){
	                p=e;
					e=0;
					for( j=0; p && p[j]; j++ ){
						if( isspace((int)p[j]) || p[j]=='>' || p[j]==':' ){
							p[j++]=0;
							while( isspace((int)p[j]) ) j++;
							e=&p[j];
							if( !*e ) e=0;
							break;
						}
					}
					if( j==0 ) p=0;
                    if( !p )goto Uf;
                }
                switch( table1[i].meth ){
                case M_STRING:
/*String: */
                    formal[i][pnum]=(char*)alloca(strlen(p)+1);
                    strcpy(formal[i][pnum],p);
                    break;
                case M_VALUE:
                    formal[i][pnum]=(char*)alloca(12);
                    key=strtoul(p,&p,16);
                    if( ((pnum==0) && table1[i].max1 && (table1[i].max1<key)) ||
                        ((pnum==1) && table1[i].max2 && (table1[i].max2<key)) )
                        strcpy(formal[i][pnum], sNA);
                    else sprintf( formal[i][pnum], "%lu", key );
                    break;
                case M_TABLE:
                    formal[i][pnum]=(char*)value_from_table(table1[i].table, strtoul(p,&p,16));
                    break;
                case M_SPECIAL:
                    switch( key ){
                    case 0: /* rev level */
                        key=strtoul(p,&p,16);
                        formal[i][pnum]=(char*)alloca(4);
                        sprintf( formal[i][pnum], "%01lx.%01lx", (key&0xFF)>>4, key&0xF );
                        break;
                    case 0x18: /* INFO0 V34/V90 bits */
                        {
                            unsigned long INFO0=strtoul(p,&p,16);
                            for( i=0; i<sizeof(table1)/sizeof(table1[0]); i++ ){
                                if( table1[i].desc==sPowerDropSupport ){
                                    formal[i][0]=(char*)((INFO0&0x40000000UL)?sON:sOFF);
                                }else if( table1[i].desc==sFreqOffset ){
                                    int off, hz2;
                                    off=(INFO0>>20)&0x3FF;
                                    formal[i][0]=(char*)alloca(8);
                                    if( off==0x3FF )strcpy( formal[i][0], sNA ); /* cpy to prevent parameter purging */
                                    else{
                                        hz2=off&0x1FF;
                                        hz2<<=1;
                                        sprintf( formal[i][0], "%s%d.%02d", (off&0x200)?"-":off?"+":"", hz2/100, hz2%100 );
                                    }
                                    off=INFO0&0x3FF;
                                    formal[i][1]=(char*)alloca(8);
                                    if( off==0x3FF )strcpy( formal[i][1], sNA ); /* cpy to prevent parameter purging */
                                    else{
                                        hz2=off&0x1FF;
                                        hz2<<=1;
                                        sprintf( formal[i][1], "%s%d.%02d", (off&0x200)?"-":off?"+":"", hz2/100, hz2%100 );
                                    }
                                }else if( table1[i].desc==sPreempIndex ){
                                    int pi;
                                    pi=(INFO0>>12)&0x0F;
                                    formal[i][0]=(char*)alloca(4);
                                    if( pi==0x0F )strcpy(formal[i][0],sNA);
                                    else sprintf( formal[i][0], "%d", pi );
                                    pi=(INFO0>>16)&0x0F;
                                    formal[i][1]=(char*)alloca(4);
                                    if( pi==0x0F )strcpy(formal[i][1],sNA);
                                    else sprintf( formal[i][1], "%d", pi );
                                }else if( table1[i].desc==sRateAsymmetry ){
                                    formal[i][0]=(char*)((INFO0&0x400)?sON:sOFF);
                                    break;
                                }
                            }
                        }
                        break;
                        /*goto String; */
                    case 0x40: /* Protocol */
                        formal[i][pnum]=(char*)value_from_table(table7, strtoul(p,&p,16));
                        break;
                    case 0x44: /* Compression */
                        formal[i][pnum]=(char*)value_from_table(table8, strtoul(p,&p,16));
                        break;
                    }
                    break;
                }
Next:           p=e;
				e=0;
				for( i=0; p && p[i]; i++ ){
					if( isspace((int)p[i]) || p[i]=='>' || p[i]==':' ){
						p[i++]=0;
						while( isspace((int)p[i]) || p[i]=='>' || p[i]==':' ) i++;
						e=&p[i];
						if( !*e ) e=0;
						break;
					}
					if( p[i]=='\"' ){
						do{ i++; } while( p[i]!='\"' && p[i] );
						if( !p[i] ) break;
					}
				}
				if( i==0 ) p=0;
            }
NextLn: continue;
        }
    }
Alez:
    for( i=0; i<sizeof(table1)/sizeof(table1[0]); i++ ){
        if( formal[i][0]!=sNA || formal[i][1]!=sNA ){
            if( table1[i].code2==P_NO ){
                sprintf( rep, "%-*s: %s", maxdesc, table1[i].desc, formal[i][0] );
                println(rep);
            }else{
                char* pad="";
                if( strchr(formal[i][0],'/') || strchr(formal[i][1],'/') )pad=" ";
                sprintf( rep, "%-*s: %s%s/%s%s", maxdesc, table1[i].desc, formal[i][0], pad, pad, formal[i][1] );
                println(rep);
            }
        }
    }
    if( !first )println("no recognizable data found!");
    return 0;
}

#ifdef _WINDOWS
static BOOL isLockClipboardUpdate=FALSE;
static BOOL isClipViewer=FALSE;
#endif

static int check_cmd_line(int ac, char *av[], char** prefix, char** file)
{
#ifdef _WINDOWS
    int clpview_specified=0;
#endif
    int prefix_specified=0;
    int i;
    for( i=0; i<ac; i++ ){
        switch( av[i][0] ){
        case '+':
            if( prefix_specified )return 0;
            *prefix=&av[i][1];
            prefix_specified=1;
            break;
        case '?':
            return 0;
        case '/':
        case '-':
            if( av[i][1]=='h' || av[i][1]=='H' || av[i][1]=='?' )return 0;
            if( stricmp(&av[i][1],"prefix")==0 ){
                if( ++i == ac )return 0;
                if( prefix_specified )return 0;
                *prefix=av[i];
                prefix_specified=1;
                break;
            }
            if( stricmp(&av[i][1],"noprefix")==0 ){
                if( prefix_specified )return 0;
                *prefix=NULL;
                prefix_specified=1;
                break;
            }
#ifdef _WINDOWS
            if( stricmp(&av[i][1],"clipview")==0 ){
                if( clpview_specified )return 0;
                isClipViewer=TRUE;
                clpview_specified=1;
                break;
            }
            if( stricmp(&av[i][1],"noclipview")==0 ){
                if( clpview_specified )return 0;
                isClipViewer=FALSE;
                clpview_specified=1;
                break;
            }
#endif
        default:
            if( *file )return 0;
            *file=av[i];
        }
    }
    return 1; /* success */
}

#ifdef _WINDOWS

static HINSTANCE hInst;
static HWND hwStatus;
static LONG PrevEditWndProc;
static char lpFileFilter[]="Log Files (*.log)\0*.log\0All Files (*.*)\0*.*\0\0";
static char* lpStartFil=NULL;

static void parse_cmdline(char *cmdstart, char **argv, char *args, int  *numargs, int  *numchars)
{
    char *p;
    int inquote;                    /* 1 = inside quotes */
    int copychar;                   /* 1 = copy char to *args */
    unsigned numslash;              /* num of backslashes seen */

    *numchars = 0;
    *numargs = 0;                   /* the program name at least */

    p = cmdstart;

    inquote = 0;

    /* loop on each argument */
    for(;;) {

        while (isspace((int)*p))++p;

        if (*p == '\0')break;   /* end of args */

        /* scan an argument */
        if (argv) *argv++ = args;     /* store ptr to arg */
        ++*numargs;

        /* loop through scanning one argument */
        for (;;) {
            copychar = 1;
            /* Rules: 2N backslashes + " ==> N backslashes and begin/end quote
               2N+1 backslashes + " ==> N backslashes + literal "
               N backslashes ==> N backslashes */
            numslash = 0;
            while (*p == '\\') { /* count number of backslashes for use below */
                ++p;
                ++numslash;
            }
            if (*p == '\"') {
            /* if 2N backslashes before, start/end quote, otherwise
                copy literally */
                if ((numslash & 1) == 0) {
                    if (inquote) {
                        if (p[1] == '\"')
                            p++;    /* Double quote inside quoted string */
                        else        /* skip first quote char and copy second */
                            copychar = 0;
                    } else copychar = 0;       /* don't copy quote */
                    inquote = !inquote;
                }
                numslash >>= 1;             /* divide numslash by two */
            }

            /* copy slashes */
            while (numslash--) {
                if (args) *args++ = '\\';
                ++*numchars;
            }

            /* if at end of arg, break loop */
            if (*p == '\0' || (!inquote && isspace((int)*p))) break;

            /* copy character into argument */
            if (copychar) {
                if (args) *args++ = *p;
                ++*numchars;
            }
            ++p;
        }

        /* null-terminate the argument */

        if (args) *args++ = '\0';          /* terminate string */
        ++*numchars;
    }
}

static void ReportError(LPCSTR prompt)
{
    MessageBox(hwe, strerror(errno), prompt, MB_OK|MB_APPLMODAL|MB_ICONERROR );
}

static void ReportW32Error(LPCSTR prompt)
{
    char buff[128];
    FormatMessage( FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), buff, sizeof(buff), NULL );
    MessageBox(hwe, buff, prompt, MB_OK|MB_APPLMODAL|MB_ICONERROR );
}

static void SaveReport(HWND hwnd)
{
    OPENFILENAME ofn;
    static char lpSaveFileName[_MAX_PATH]="";
    memset(&ofn,0,sizeof(ofn));
    ofn.lStructSize=sizeof(ofn);
    ofn.hwndOwner=hwnd;
    ofn.lpstrFilter=lpFileFilter;
    ofn.nFilterIndex=1;
    ofn.lpstrFile=lpSaveFileName;
    ofn.nMaxFile=sizeof(lpSaveFileName)/sizeof(lpSaveFileName[0]);
    ofn.Flags=OFN_NOREADONLYRETURN|OFN_OVERWRITEPROMPT|OFN_PATHMUSTEXIST|OFN_EXTENSIONDIFFERENT;
    ofn.lpstrDefExt="log";
    if( GetSaveFileName(&ofn) ){
        HANDLE hFile;
        DWORD dwLen=SendMessage(hwe,WM_GETTEXTLENGTH,0,0);
        char *buff=(TCHAR*)alloca((dwLen+1)*sizeof(buff[0]));
        SendMessage(hwe,WM_GETTEXT,dwLen+1,(LPARAM)buff);
        hFile=CreateFile(lpSaveFileName,GENERIC_WRITE,FILE_SHARE_READ,0,CREATE_ALWAYS,FILE_ATTRIBUTE_NORMAL,0);
        if( hFile==INVALID_HANDLE_VALUE )ReportW32Error("Cannot create file");
        else{
            DWORD written;
            BOOL err=TRUE;
            err&=WriteFile(hFile,prefix,strlen(prefix),&written,0);
            err&=WriteFile(hFile,Logo1,sizeof(Logo1)-1,&written,0);
            err&=WriteFile(hFile,crlf,sizeof(crlf)-1,&written,0);
            err&=WriteFile(hFile,prefix,strlen(prefix),&written,0);
            err&=WriteFile(hFile,Logo2,sizeof(Logo2)-1,&written,0);
            err&=WriteFile(hFile,crlf,sizeof(crlf)-1,&written,0);
            err&=WriteFile(hFile,prefix,strlen(prefix),&written,0);
            err&=WriteFile(hFile,Logo3,sizeof(Logo3)-1,&written,0);
            err&=WriteFile(hFile,crlf,sizeof(crlf)-1,&written,0);
            if( buff[0]!='\0' )err&=WriteFile(hFile,buff,strlen(buff),&written,0);
            if( !err )ReportW32Error("cannot save file");
            CloseHandle(hFile);
        }
    }
}

static void LoadStatsFromFile(const char* lpOpenFileName)
{
    FILE* hFile=_fsopen(lpOpenFileName,"rt",SH_DENYWR);
    if( !hFile )ReportError("Cannot open file");
    else{
        edit_buff[0]='\0';
        MakeStat(hFile);
        fclose(hFile);
        SendMessage(hwe,WM_SETTEXT,0,(LPARAM)edit_buff);
        SendMessage(hwStatus,SB_SETTEXT,0,(LPARAM)lpOpenFileName);
    }
}

static void LoadStats(HWND hwnd)
{
    OPENFILENAME ofn;
    static char lpOpenFileName[_MAX_PATH]="";
    memset(&ofn,0,sizeof(ofn));
    ofn.lStructSize=sizeof(ofn);
    ofn.hwndOwner=hwnd;
    ofn.lpstrFilter=lpFileFilter;
    ofn.nFilterIndex=1;
    ofn.lpstrFile=lpOpenFileName;
    ofn.nMaxFile=sizeof(lpOpenFileName)/sizeof(lpOpenFileName[0]);
    ofn.Flags=OFN_PATHMUSTEXIST|OFN_FILEMUSTEXIST|OFN_EXTENSIONDIFFERENT|OFN_READONLY;
    ofn.lpstrDefExt="log";
    if( GetOpenFileName(&ofn) )LoadStatsFromFile(lpOpenFileName);
}

static int CALLBACK AboutDlgProc(HWND hwnd,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
    (void)lParam;
    switch(uMsg){
    case WM_INITDIALOG:
        SendMessage(GetDlgItem(hwnd,IDC_STATIC_ABOUT1),WM_SETTEXT,0,(LPARAM)Logo1);
        SendMessage(GetDlgItem(hwnd,IDC_STATIC_ABOUT2),WM_SETTEXT,0,(LPARAM)"Copyright (c) 2000-2002, Stanislav V. Mekhanoshin");
        SendMessage(GetDlgItem(hwnd,IDC_STATIC_ABOUT3),WM_SETTEXT,0,(LPARAM)"Mail-to: rampitec@tu.spb.ru, 2:5030/172.9@fidonet");
        return TRUE ;
    case WM_COMMAND:
        if( wParam!=IDOK && wParam!=IDCANCEL )break;
    case WM_CLOSE:
        EndDialog(hwnd,IDOK);
        return TRUE;
    }
    return FALSE;
}

static int CALLBACK EditWndProc(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam )
{
    switch(uMsg){
    case WM_PASTE:
        SendMessage(GetParent(hwnd),WM_COMMAND,ID_FILE_PROCESSCLIPBOARD,(LPARAM)hwnd);
        return TRUE;
    case WM_COPY:
        if( OpenClipboard(hwnd) ){
            HGLOBAL hGlb=NULL;
            DWORD dwStart,dwEnd;
            if( EmptyClipboard() ){
                SendMessage(hwe,EM_GETSEL,(WPARAM)&dwStart,(LPARAM)&dwEnd);
                hGlb=GlobalAlloc(GHND,sizeof(Logo1)+sizeof(Logo2)+sizeof(Logo3)+7*sizeof(Logo1[0])+dwEnd-dwStart);
                if( hGlb ){
                    DWORD l;
                    char *p=(char*)GlobalLock(hGlb);
                    strcpy(p,Logo1);
                    strcat(p,"\r\n");
                    strcat(p,Logo2);
                    strcat(p,"\r\n");
                    strcat(p,Logo3);
                    strcat(p,"\r\n");
                    l=strlen(p);
                    strncat(&p[l],&edit_buff[dwStart],dwEnd-dwStart);
                    p[l+dwEnd-dwStart]='\0';
                    GlobalUnlock(hGlb);
                    if( !SetClipboardData(CF_TEXT,hGlb) ){
                        GlobalFree(hGlb);
                        hGlb=NULL;
                    }else isLockClipboardUpdate=isClipViewer;
                }
            }
            CloseClipboard();
            if( hGlb )GlobalFree(hGlb);
            return TRUE;
        }
    }
    return CallWindowProc((WNDPROC)PrevEditWndProc,hwnd,uMsg,wParam,lParam);
}

#define BK_COLOR COLOR_WINDOW

#ifndef SB_SETICON
#define SB_SETICON              (WM_USER+15)
#endif
#if defined(__RSXNT__) || defined(__WATCOMC__)
typedef struct tagINITCOMMONCONTROLSEX {
    DWORD dwSize;             /* size of this structure */
    DWORD dwICC;              /* flags indicating which classes to be initialized */
} INITCOMMONCONTROLSEX, *LPINITCOMMONCONTROLSEX;
#define ICC_BAR_CLASSES      0x00000004 /* toolbar, statusbar, trackbar, tooltips */
#if defined(__RSXNT__)
WINBOOL STDCALL InitCommonControlsEx(LPINITCOMMONCONTROLSEX);
#elif defined(__WATCOMC__)
WINCOMMCTRLAPI BOOL WINAPI InitCommonControlsEx(LPINITCOMMONCONTROLSEX);
#endif
typedef struct tagNMMOUSE {
    NMHDR   hdr;
    DWORD   dwItemSpec;
    DWORD   dwItemData;
    POINT   pt;
    DWORD   dwHitInfo; /* any specifics about where on the item or control the mouse is */
} NMMOUSE, FAR* LPNMMOUSE;
#endif /* __RSXNT__ || __WATCOMC_ */

typedef HANDLE (__stdcall * LoadImageProc)(HINSTANCE,LPCSTR,UINT,int,int,UINT);

int CALLBACK ShowDlgProc(HWND hwndDlg, UINT uMsg, WPARAM wParam, LPARAM lParam )
{
    static COLORREF clBkEdit;
    static HMENU hwMenu;
    static HMENU hwMenuFile;
    static HBRUSH hBkBrush=NULL;
    static HWND hwndNextClip=NULL;
    static INT aSbWidth[2]={0,-1};
    static const char sClipViewOff[]="Clipboard Viewer Off";
    static const char sClipViewOn[]="Clipboard Viewer On";
    static HICON hiClipViewOff=NULL;
    static HICON hiClipViewOn=NULL;
    switch(uMsg){
    case WM_INITDIALOG:
        {
            HANDLE hUser32;
            LoadImageProc prc;
            HICON hIcon=LoadIcon(hInst,MAKEINTRESOURCE(ID_UD));
            SendMessage(hwndDlg,WM_SETICON,ICON_SMALL,(LPARAM)hIcon);
            SendMessage(hwndDlg,WM_SETICON,ICON_BIG  ,(LPARAM)hIcon);
            hwe=GetDlgItem(hwndDlg,IDC_EDIT_STAT);
            PrevEditWndProc=GetWindowLong(hwe,GWL_WNDPROC);
            SetWindowLong(hwe,GWL_WNDPROC,(LONG)EditWndProc);
            hwStatus=CreateStatusWindow(CCS_BOTTOM|SBARS_SIZEGRIP|WS_CHILD|WS_VISIBLE,"",hwndDlg,IDC_STATUSBAR);
            hUser32=GetModuleHandle("USER32.DLL");
            if( hUser32 ){
                prc=(LoadImageProc)GetProcAddress(hUser32,"LoadImageA");
                if( prc ){
                    hiClipViewOn=(HICON)prc(hInst,MAKEINTRESOURCE(IDI_CLPVIEW_ON),IMAGE_ICON,0,0,LR_DEFAULTCOLOR);
                    hiClipViewOff=(HICON)prc(hInst,MAKEINTRESOURCE(IDI_CLPVIEW_OFF),IMAGE_ICON,0,0,LR_DEFAULTCOLOR);
                }
            }
            hwMenu=GetMenu(hwndDlg);
            hwMenuFile=GetSubMenu(hwMenu,0);
            clBkEdit=GetSysColor(BK_COLOR);
            hBkBrush=GetSysColorBrush(BK_COLOR);
            if( isClipViewer ){
                isClipViewer=FALSE;
                SendMessage(hwndDlg,WM_COMMAND,ID__CLIPVIEWER,(LPARAM)hwndDlg);
            }
            if( !isClipViewer && !lpStartFil )PostMessage(hwndDlg,WM_COMMAND,ID_FILE_PROCESSCLIPBOARD,(LPARAM)hwndDlg);
            if( lpStartFil ){
                LoadStatsFromFile(lpStartFil);
            }
            PostMessage(hwe,EM_SETSEL,0,0);
        }
        return TRUE;
    case WM_SYSCOLORCHANGE:
        clBkEdit=GetSysColor(BK_COLOR);
        hBkBrush=GetSysColorBrush(BK_COLOR);
        return FALSE;
    case WM_CTLCOLOREDIT:
    case WM_CTLCOLORSTATIC:
        if( hwe!=(HWND)lParam )return FALSE;
        SetBkColor((HDC)wParam,clBkEdit);
        return (BOOL)hBkBrush;
    case WM_CLOSE:
        DestroyWindow(hwndDlg);
        return TRUE;
    case WM_SIZE:
        {
            int x=LOWORD(lParam);
            int y=HIWORD(lParam);
            RECT rc={0,0,0,0};
            GetWindowRect(hwStatus,&rc);
            MoveWindow(hwStatus,0,y-(rc.bottom-rc.top),x,rc.bottom-rc.top,TRUE);
            MoveWindow(GetDlgItem(hwndDlg,IDC_EDIT_STAT),0,0,x,y-(rc.bottom-rc.top),TRUE);
            aSbWidth[0]=x-150;
            if( aSbWidth[0]<50 )aSbWidth[0]=50;
            SendMessage(hwStatus,SB_SETPARTS,sizeof(aSbWidth)/sizeof(aSbWidth[0]),(LPARAM)aSbWidth);
            SendMessage(hwStatus,SB_SETTEXT,1,(LPARAM)(isClipViewer?sClipViewOn:sClipViewOff));
            SendMessage(hwStatus,SB_SETICON,1,(LPARAM)(isClipViewer?hiClipViewOn:hiClipViewOff));
        }
        return FALSE;
    case WM_DROPFILES:
        {
            char lpFile[_MAX_PATH+1]="";
            DragQueryFile((HDROP)wParam,0,lpFile,sizeof(lpFile)/sizeof(lpFile[0]));
            LoadStatsFromFile(lpFile);
            DragFinish((HDROP)wParam);
        }
        return TRUE;
    case WM_MENUSELECT:
        {
            typedef struct tagSTATUSSTRING{
                HMENU hMenu;
                UINT uiString;
            }STATUS_STRING;
            static STATUS_STRING StatusStr[1];
            MenuHelp(uMsg,wParam,lParam,hwMenu,hInst,hwStatus,(UINT*)StatusStr);
        }
        return FALSE;
    case WM_COMMAND:
        switch(LOWORD(wParam)){
        case IDCLOSE:
            DestroyWindow(hwndDlg);
            return TRUE;
        case ID_FILE_PROCESSCLIPBOARD:
            edit_buff[0]='\0';
            MakeStat(NULL);
            SendMessage(hwe,WM_SETTEXT,0,(LPARAM)edit_buff);
            SendMessage(hwStatus,SB_SETTEXT,0,(LPARAM)"Report from clipboard");
            return TRUE;
        case ID_FILE_SAVEREPORT:
            SaveReport(hwndDlg);
            return TRUE;
        case ID_FILE_LOADLOGFILE:
            LoadStats(hwndDlg);
            return TRUE;
        case ID_EDIT_SELECTALL:
            SendMessage(hwe,EM_SETSEL,0,-1);
            return TRUE;
        case ID_HELP_ABOUT:
            DialogBox(hInst,MAKEINTRESOURCE(IDD_DIALOG_ABOUT),hwndDlg,AboutDlgProc);
            return TRUE;
        case ID__CLIPVIEWER:
            if( !isClipViewer ){
                hwndNextClip=SetClipboardViewer(hwndDlg);
                isClipViewer=TRUE;
                CheckMenuItem(hwMenuFile,ID__CLIPVIEWER,MF_BYCOMMAND|MF_CHECKED);
                SendMessage(hwStatus,SB_SETTEXT,1,(LPARAM)sClipViewOn);
                SendMessage(hwStatus,SB_SETICON,1,(LPARAM)hiClipViewOn);
            }else{
                ChangeClipboardChain(hwndDlg,hwndNextClip);
                hwndNextClip=NULL;
                isClipViewer=FALSE;
                CheckMenuItem(hwMenuFile,ID__CLIPVIEWER,MF_BYCOMMAND|MF_UNCHECKED);
                SendMessage(hwStatus,SB_SETTEXT,1,(LPARAM)sClipViewOff);
                SendMessage(hwStatus,SB_SETICON,1,(LPARAM)hiClipViewOff);
            }
            return TRUE;
        }
        break;
    case WM_CHANGECBCHAIN:
        if( (HWND)wParam==hwndNextClip )hwndNextClip=(HWND)lParam;
        else if( hwndNextClip )SendMessage(hwndNextClip,WM_CHANGECBCHAIN,wParam,lParam);
        return 0;
    case WM_DRAWCLIPBOARD:
        if( !isLockClipboardUpdate )SendMessage(hwndDlg,WM_COMMAND,ID_FILE_PROCESSCLIPBOARD,(LPARAM)hwndDlg);
        isLockClipboardUpdate=FALSE;
        if( hwndNextClip && hwndNextClip!=hwndDlg )SendMessage(hwndNextClip,WM_DRAWCLIPBOARD,wParam,lParam);
        return 0;
    case WM_NOTIFY:
        if( ((LPNMHDR)lParam)->hwndFrom!=hwStatus || ((LPNMHDR)lParam)->code!=NM_CLICK )return FALSE;
        if( ((LPNMMOUSE)lParam)->dwItemSpec==1 ){
            SendMessage(hwndDlg,WM_COMMAND,ID__CLIPVIEWER,(LPARAM)hwndDlg);
            return TRUE;
        }
        return FALSE;
    case WM_DESTROY:
        PostQuitMessage(0);
        if( isClipViewer ){
            ChangeClipboardChain(hwndDlg,hwndNextClip);
            hwndNextClip=NULL;
            isClipViewer=FALSE;
        }
        DestroyMenu(hwMenuFile); hwMenuFile=NULL;
        DestroyMenu(hwMenu); hwMenu=NULL;
        DestroyIcon(hiClipViewOn);
        DestroyIcon(hiClipViewOff);
        return 0;
    }
    return FALSE;
}

int WINAPI
#ifdef _NO_COM
    WinMain
#else
    WinMainAlone
#endif
                  ( HINSTANCE hInstance,
                    HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine,
                    int nCmdShow )
{
    static INITCOMMONCONTROLSEX initctl={sizeof(INITCOMMONCONTROLSEX),ICC_BAR_CLASSES};
    char *args;
    int numargs;
    int numchars;
    char *pr;
    HWND hwnd;
    MSG  msg;
    HACCEL hAccel;
    char sProfile[_MAX_PATH];
    char sDefPrefix[80];
    char sDefFile[_MAX_PATH];

    (void)nCmdShow;
    (void)hPrevInstance;
    prefix="";
    hInst=hInstance;
    parse_cmdline(lpszCmdLine,NULL,NULL,&numargs,&numchars);
    args = (char*)alloca(numargs * sizeof(char*) + numchars * sizeof(char)+1);
    parse_cmdline(lpszCmdLine,(char**)args,args+sizeof(char*)*numargs,&numargs,&numchars);
    pr=NULL;
    lpStartFil=NULL;
    GetModuleFileName(hInstance,sProfile,sizeof(sProfile));
    strcpy(strrchr(sProfile,'.'),".ini");
    isClipViewer=((GetPrivateProfileInt("Default","ClipboardViewer",0,sProfile)>0)?TRUE:FALSE);
    if( !check_cmd_line(numargs,(char**)args,&pr,&lpStartFil ) ){
        MessageBox( HWND_DESKTOP, "Usage: #udw [+prefix | -prefix string | -noprefix] [-clipview | -noclipview] [file]", "#udw", MB_OK|MB_SETFOREGROUND|MB_ICONINFORMATION );
        return 0; /* term before message loop; */
    }
    if( !pr ){
        GetPrivateProfileString("Default","Prefix","",sDefPrefix,sizeof(sDefPrefix),sProfile);
        if( sDefPrefix[0] )pr=sDefPrefix;
    }
    if( !lpStartFil ){
        GetPrivateProfileString("Default","OpenFileName","",sDefFile,sizeof(sDefFile),sProfile);
        if( sDefFile[0] )lpStartFil=sDefFile;
    }
    if( pr ){
        prefix=(char*)alloca(strlen(pr)+2);
        strcpy(prefix,pr);
        strcat(prefix," ");
    }
    edit_buff=HeapAlloc(GetProcessHeap(),HEAP_GENERATE_EXCEPTIONS|HEAP_ZERO_MEMORY,INIT_EDIT_BUFF_SIZE);
    if( !edit_buff ){
        ReportW32Error("memory alocation failed");
        return 1;
    }
    SetLastError(ERROR_SUCCESS);
    if( !InitCommonControlsEx(&initctl) )ReportW32Error("cannot init common controls");
    hwnd=CreateDialog(hInstance,MAKEINTRESOURCE(ID_UD),HWND_DESKTOP,ShowDlgProc);
    if( !hwnd ){
        ReportW32Error("cannot create window");
        return 0;
    }
    hAccel=LoadAccelerators(hInst,MAKEINTRESOURCE(IDR_ACCELERATOR));
    while(GetMessage(&msg, NULL, 0, 0))
    {
        if( TranslateAccelerator(hwnd,hAccel,&msg) )continue;
        if( IsDialogMessage(hwnd,&msg) )continue;
        TranslateMessage(&msg) ;
        DispatchMessage(&msg) ;
    }
    HeapFree(GetProcessHeap(),0,edit_buff);
    DestroyWindow(hwnd);
    return msg.wParam ;
}

#else /* _WINDOWS */

int __cdecl main(int ac, char *av[])
{
    char* pr, *fil;
    FILE* hIn=stdin;
    int i;
    prefix="";
    pr=NULL;
    fil=NULL;
    if( !check_cmd_line(ac-1,&av[1],&pr,&fil) ){
#if !defined(__CLIPBOARD__) || defined(__RSXNT__)
Usage:
#endif
#if defined(__CLIPBOARD__)
        println( "Usage: #ud [+prefix | -prefix string | -noprefix] [<stream | file]" );
#else
        println( "Usage: #ud [+prefix | -prefix string | -noprefix] {<stream | file}" );
#endif /* __CLIPBOARD__ */
        return 1;
    }
    if( pr ){
        prefix=(char*)alloca(strlen(pr)+2);
        strcpy(prefix,pr);
        strcat(prefix," ");
    }
    if( fil ){
        hIn=_fsopen(fil,"rt",SH_DENYWR);
        if( !hIn ){
            printf( "cannot open file %s: %s\n", fil, strerror(errno) );
            return 1;
        }
    }
#if !defined(__CLIPBOARD__)
    if( isdevice(hIn) )goto Usage;
#elif defined(__RSXNT__)
    if( !IsW32 && isdevice(hIn) )goto Usage;
#endif /* !__CLIPBOARD__ */
    println( Logo1 );
    println( Logo2 );
    println( Logo3 );
    i=MakeStat(hIn);
    if( hIn && fil )fclose(hIn);
    return i;
}
#endif /* !_WINDOWS */
